<?php
namespace RawMaterial\Controllers;

use User;
use DateTime;
use Grocery_CRUD;
use Tendoo_Module;
use RawMaterial\Classes\Material;

class SuppliesController extends Tendoo_Module
{
    public function __construct()
    {
        parent::__construct();
    }

    public function suppliesCrud( $page = '', $id = '' )
    {
        if ( $page === 'add' ) {
            
            $material       =   new Material;
            $units      =   $material->getUnits();
            $materials  =   $material->getMaterials();
            $supplies   =   $material->getSupplies();

            $this->Gui->set_title( store_title( __( 'New Material Procurement', 'raw-material' ) ) );
            $this->load->module_view( 'raw-material', 'supplies.gui', compact( 'units', 'materials', 'supplies' ) );
        } else if ( $page === 'items' ) {
            $material       =   new Material;
            $supply         =   $material->getSupply( $id );
            $units          =   $material->getUnits();
            $materials      =   $material->getMaterials();

            $this->Gui->set_title( 
                store_title( 
                    sprintf( 
                        __( '%s > Products', 'raw-material' ), 
                        $supply[ 'NAME' ] 
                    ) 
                )
            );

            $this->load->module_view( 'raw-material', 'supplies.items.gui', compact( 'supply', 'units', 'materials' ) );
        } else if ( $page === 'refresh' ) {
            $this->load->library('user_agent');
            $material       =   new Material;
            $material->refreshProcurement( $id );
            return redirect( site_url([ 'dashboard', store_slug(), 'raw-material', 'supplies' ]) );
        } else {
            $data[ 'crud_content' ]      =   $this->suppliesCrudHead();
            $this->Gui->set_title( store_title( __( 'Raw Material Supplies', 'raw-material' ) ) );
            $this->load->module_view( 'raw-material', 'common.gui', $data );
        }
    }

    public function suppliesCrudHead()
    {
		/**
		 * This feature is not more accessible on main site when
		 * multistore is enabled
		**/
		
		if( ( multistore_enabled() && ! is_multistore() ) ) {
			return show_error( __( 'This feature doesn\'t works on the warehouse.', 'raw-material' ) );
		}
        
        $crud = new Grocery_CRUD();

        $crud->set_theme('bootstrap');
        $crud->set_subject(__('Material Supplies', 'raw-material' ));
        $crud->set_table( $this->db->dbprefix( store_prefix() . 'gastro_materials_supplies' ) );
		
		$fields					=	array( 'NAME', 'DESCRIPTION', 'AUTHOR', 'DATE_CREATION', 'DATE_MOD' );
		$crud->columns( 'NAME', 'TOTAL_ITEMS', 'TOTAL_COST', 'AUTHOR', 'DATE_MOD' );
        $crud->fields( $fields );
        
        $state = $crud->getState();

        /**
         * Callback to support date formating
         * @since 3.12.8
         */
        $crud->callback_column( 'DATE_CREATION', function( $date ) {
            $datetime   =    new DateTime( $date ); 
            return $datetime->format( store_option( 'nexo_datetime_format', 'Y-m-d h:i:s' ) );
        });
        $crud->callback_column( 'DATE_MOD', function( $date ) {
            $datetime   =    new DateTime( $date ); 
            return $datetime->format( store_option( 'nexo_datetime_format', 'Y-m-d h:i:s' ) );
        });

        $crud->callback_column( 'TOTAL_COST', function( $amount ) {
            return get_instance()->Nexo_Misc->cmoney_format( $amount );
        });

        $crud->display_as( 'NAME',              __( 'Supply Name', 'raw-material' ) );
        $crud->display_as( 'TOTAL_COST',        __( 'Cost', 'raw-material' ) );
        $crud->display_as( 'TOTAL_ITEMS',       __( 'Items Included', 'raw-material' ) );
        $crud->display_as( 'DESCRIPTION',       __( 'Descrpition', 'raw-material' ) );
        $crud->display_as( 'AUTHOR',            __( 'Author', 'raw-material' ) );
        $crud->display_as( 'DATE_CREATION',     __( 'Created On', 'raw-material' ) );
        $crud->display_as( 'DATE_MOD',          __( 'Modified On', 'raw-material' ) );

        $crud->callback_before_insert( function( $data ) {
            $data[ 'AUTHOR' ]           =   User::id();
            $data[ 'DATE_CREATION' ]    =   date_now();
            $data[ 'DATE_MOD' ]         =   date_now();
            return $data;
        });

        $crud->callback_before_update( function( $data ) {
            $data[ 'AUTHOR' ]           =   User::id();
            $data[ 'DATE_MOD' ]         =   date_now();
            return $data;
        });

        $crud->callback_before_delete( function( $id ) {
            $material   =   new Material;
            $material->deleteProcurementEntries( $id );
        });


        $crud->add_action(
			__( 'See Procured Material', 'raw-material'),
            '',
            '',
			'fa fa-eye',
			[ $this, 'getAllUnitLink' ]
        );

        $crud->add_action(
			__( 'Refresh', 'raw-material'),
            '',
            '',
			'fa fa-refresh',
			[ $this, 'gotRefreshProcurement' ]
        );
        
        // XSS Cleaner
        $this->events->add_filter('grocery_callback_insert', array( $this->grocerycrudcleaner, 'xss_clean' ));
        $this->events->add_filter('grocery_callback_update', array( $this->grocerycrudcleaner, 'xss_clean' ));
        
        $crud->required_fields('TTILE');

        $crud->change_field_type('DATE_CREATION', 'invisible');
        $crud->change_field_type('DATE_MOD', 'invisible');
        $crud->change_field_type('AUTHOR', 'invisible');

        $crud->set_relation('AUTHOR', 'aauth_users', 'name');
        
        $crud->unset_jquery();
        $output = $crud->render();
        
        foreach ($output->js_files as $files) {
            $this->enqueue->js(substr($files, 0, -3), '');
        }
        foreach ($output->css_files as $files) {
            $this->enqueue->css(substr($files, 0, -4), '');
        }
        
        return $output;
    }

    public function getAllUnitLink( $id ) 
    {
        return site_url(array( 'dashboard', store_slug(), 'raw-material', 'supplies', 'items', $id )) . '/';
    }

    public function gotRefreshProcurement( $id, $row )
    {
        return site_url([ 'dashboard', store_slug(), 'raw-material', 'supplies', 'refresh', $id ]);
    }

    public function unitsCrudHead()
    {
		/**
		 * This feature is not more accessible on main site when
		 * multistore is enabled
		**/
		
		if( ( multistore_enabled() && ! is_multistore() ) ) {
			return show_error( __( 'This feature doesn\'t works on the warehouse.', 'raw-material' ) );
		}
        
        $crud = new Grocery_CRUD();

        $crud->set_theme('bootstrap');
        $crud->set_subject(__('Material Units', 'raw-material' ));
        $crud->set_table( $this->db->dbprefix( store_prefix() . 'gastro_materials_units' ) );
		
		$fields					=	array( 'NAME', 'REF_UNIT_GROUP', 'UNIT_NAME', 'BASE_UNIT', 'UNIT_VALUE', 'DESCRIPTION', 'AUTHOR', 'DATE_CREATION', 'DATE_MOD' );
		$crud->columns( 'NAME', 'UNIT_NAME', 'BASE_UNIT', 'UNIT_VALUE', 'REF_UNIT_GROUP', 'DESCRIPTION', 'DATE_MOD' );
        $crud->fields( $fields );
        
        $state = $crud->getState();
        
        /**
         * Callback to support date formating
         * @since 3.12.8
         */
        $crud->callback_column( 'DATE_CREATION', function( $date ) {
            $datetime   =    new DateTime( $date ); 
            return $datetime->format( store_option( 'nexo_datetime_format', 'Y-m-d h:i:s' ) );
        });
        $crud->callback_column( 'DATE_MOD', function( $date ) {
            $datetime   =    new DateTime( $date ); 
            return $datetime->format( store_option( 'nexo_datetime_format', 'Y-m-d h:i:s' ) );
        });
        
        $crud->callback_column( 'BASE_UNIT', function( $value ) {
            return ( bool ) $value ? __( 'Yes', 'raw-material' ) : __( 'No', 'raw-material' );
        });
        
        
        $crud->display_as( 'NAME',              __( 'Unit Name', 'raw-material' ) );
		$crud->display_as( 'UNIT_NAME',         __( 'Unit Code', 'raw-material'  ) );
        $crud->display_as( 'UNIT_VALUE',        __( 'Unit Value', 'raw-material' ) );
		$crud->display_as( 'BASE_UNIT',         __( 'Base Unit', 'raw-material'  ) );
		$crud->display_as( 'REF_UNIT_GROUP',    __( 'Unit Group', 'raw-material'  ) );
        $crud->display_as( 'DESCRIPTION',       __( 'Descrpition', 'raw-material' ) );
        $crud->display_as( 'AUTHOR',            __( 'Author', 'raw-material' ) );
        $crud->display_as( 'DATE_CREATION',     __( 'Created On', 'raw-material' ) );

        $crud->field_description( 'UNIT_VALUE', __( 'This value should be based on the base unit, if this entity is not a base unit. It should be considered as multiplicator others unit (not for the base unit). E.g if a cup is 2.5ml (base unit) a bottle could be 10 time a cup (10 x 2.5).', 'raw-material' ) );
        $crud->field_description( 'UNIT_NAME', __( 'For example : kg, grams, l, liter, pack, piece', 'raw-material' ) );
        $crud->field_description( 'BASE_UNIT', __( 'This entity will be used as the base reference for the assigned group.', 'raw-material' ) );
        $crud->field_description( 'REF_UNIT_GROUP', __( 'A unit entity need to be assigned to a unit group.', 'raw-material' ) );

        $crud->callback_before_insert( function( $data ) {
            $data[ 'AUTHOR' ]           =   User::id();
            $data[ 'DATE_CREATION' ]    =   date_now();
            $data[ 'DATE_MOD' ]         =   date_now();

            return $data;
        });

        $crud->callback_before_update( function( $data ) {
            $data[ 'AUTHOR' ]           =   User::id();
            $data[ 'DATE_MOD' ]         =   date_now();
            return $data;
        });

        $crud->callback_after_insert( function( $data, $id ) {
            $material                   =   new Material;

            if ( ( bool ) $data[ 'BASE_UNIT' ] === true ) {
                $material->switchBaseUnit( $id );
            }
        });

        $crud->callback_after_update( function( $data, $id ) {
            $material                   =   new Material;

            if ( ( bool ) $data[ 'BASE_UNIT' ] === true ) {
                $material->switchBaseUnit( $id );
            }
        });
        
        // XSS Cleaner
        $this->events->add_filter('grocery_callback_insert', array( $this->grocerycrudcleaner, 'xss_clean' ));
        $this->events->add_filter('grocery_callback_update', array( $this->grocerycrudcleaner, 'xss_clean' ));
        
        $crud->required_fields([ 
            'NAME',
            'UNIT_NAME',
            'REF_UNIT_GROUP',
            'BASE_UNIT',
        ]);

        $crud->change_field_type('DATE_CREATION', 'invisible');
        $crud->change_field_type('DATE_MOD', 'invisible');
        $crud->change_field_type('AUTHOR', 'invisible');
        $crud->change_field_type('BASE_UNIT', 'dropdown', [
            1   =>  __( 'Yes', 'raw-material' ),
            0   =>  __( 'No', 'raw-material' ),
        ]);

        $crud->set_relation('AUTHOR', 'aauth_users', 'name');
        $crud->set_relation('REF_UNIT_GROUP', store_prefix() . 'gastro_materials_units_groups', 'NAME' );
        
        $crud->unset_jquery();
        $output = $crud->render();
        
        foreach ($output->js_files as $files) {
            $this->enqueue->js(substr($files, 0, -3), '');
        }
        foreach ($output->css_files as $files) {
            $this->enqueue->css(substr($files, 0, -4), '');
        }
        
        return $output;
    }
}