<?php
namespace RawMaterial\Controllers;

use User;
use DateTime;
use Grocery_CRUD;
use Tendoo_Module;
use RawMaterial\Classes\Material;

class MaterialController extends Tendoo_Module
{
    public function getMaterialCrud( $page = null, $id = null )
    {
        if ( $page === 'quantities' ) {
            $material           =   new Material;
            $currentMaterial    =   $material->getMaterial( $id );

            if ( empty( $currentMaterial ) ) {
                return show_404( __( 'Unable to find the material you\'re looking for.', 'raw-material' ) );
            }

            $materialStock      =   $material->getMaterialStock( $id );
            $units              =   $material->getUnits();
            $baseUnit           =   $material->getGroupBaseUnit( $currentMaterial[ 'REF_UNIT_GROUP' ] );

            if ( empty( $baseUnit ) ) {
                return show_error( sprintf( 
                    __( 'This unit does\'nt any base unit defined. <a href="%s">Please define one</a>.', 'raw-material' ),
                    site_url([ 'dashboard', store_slug(), 'raw-material', 'units', 'add' ])
                ) );
            }
            
            $this->Gui->set_title( store_title( sprintf( __(
                'Material Unit Quantities : %s', 'raw-material'
            ), $currentMaterial[ 'NAME' ] ) ) );

            $this->load->module_view( 'raw-material', 'material-stock.gui', compact( 'materialStock', 'units', 'currentMaterial', 'baseUnit' ) );
        } else {
            $data[ 'crud_content' ]    	=   $this->materialCrudHeader();
            $this->Gui->set_title( store_title( __( 'Materials', 'raw-material' ) ) );
            $this->load->module_view( 'raw-material', 'common.gui', $data );
        }
    }

    public function getAdjustmentCrud( $page = null, $id = null )
    {
        if ( $page === 'new' ) {
            $this->Gui->set_title( store_title( __( 'New Adjustment', 'raw-material' ) ) );
            
            $materialService    =   new Material;
            $materials          =   $materialService->getMaterials();
            $actions            =   [
                [
                    'label'     =>  __( 'Deduct' ),
                    'value'     =>  'deduct',
                ], [
                    'label'     =>  __( 'Increase' ),
                    'value'     =>  'increase',
                ]
            ];

            $this->load->module_view( 'raw-material', 'materials.adjustment.gui', compact( 'materials', 'actions' ) );

        } else {
            $data[ 'crud_content' ]    	=   $this->materialAdjustmentCrudHeader();
            $this->Gui->set_title( store_title( __( 'Material Adjustments', 'raw-material' ) ) );
            $this->load->module_view( 'raw-material', 'common.gui', $data );
        }
    }

    /**
     * Build the crud header that
     * handle the material adjustment
     * @return array
     */
    public function materialAdjustmentCrudHeader()
    {
        /**
		 * This feature is not more accessible on main site when
		 * multistore is enabled
		**/
		
		if( ( multistore_enabled() && ! is_multistore() ) ) {
			return show_error( __( 'This feature doesn\'t works on the warehouse.', 'raw-material' ) );
		}
        
        $crud = new Grocery_CRUD();

        $crud->set_theme('bootstrap');
        $crud->set_subject(__('Materials Adjustments', 'raw-material' ));
        $crud->set_table( $this->db->dbprefix( store_prefix() . 'gastro_materials_adjustments' ) );
		
		$fields					=	[
            'REF_MATERIAL',
            'REF_UNIT',
            'ACTION',
            'VALUE',
            'AUTHOR',
            'DESCRIPTION',
        ];

		$crud->columns( 'ACTION', 'REF_UNIT', 'VALUE', 'REF_MATERIAL', 'AUTHOR', 'DATE_MOD' );
        $crud->fields( $fields );
        
        $state = $crud->getState();
        
        /**
         * Callback to support date formating
         * @since 3.12.8
         */
        $crud->callback_column( 'DATE_CREATION', function( $date ) {
            $datetime   =    new DateTime( $date ); 
            return $datetime->format( store_option( 'nexo_datetime_format', 'Y-m-d h:i:s' ) );
        });
        
        $crud->callback_column( 'DATE_MOD', function( $date ) {
            $datetime   =    new DateTime( $date ); 
            return $datetime->format( store_option( 'nexo_datetime_format', 'Y-m-d h:i:s' ) );
        });

        $crud->display_as( 'REF_MATERIAL',              __( 'Material', 'raw-material' ) );
        $crud->display_as( 'REF_UNIT',    __( 'Unit', 'raw-material' ) );
        $crud->display_as( 'ACTION',    __( 'Action', 'raw-material' ) );
        $crud->display_as( 'VALUE',    __( 'Value', 'raw-material' ) );
        $crud->display_as( 'DESCRIPTION',       __( 'Descrpition', 'raw-material' ) );
        $crud->display_as( 'AUTHOR',            __( 'Author', 'raw-material' ) );
        $crud->display_as( 'DATE_CREATION',     __( 'Created On', 'raw-material' ) );
        $crud->display_as( 'DATE_MOD',          __( 'Modified On', 'raw-material' ) );

        $crud->field_type( 'ACTION', 'dropdown', [
            'deduct'        =>      __( 'Deducting', 'raw-material' ),
            'increase'      =>      __( 'Increase', 'raw-material' ),
        ]);

        $crud->callback_before_insert( function( $data ) {
            $data[ 'AUTHOR' ]           =   User::id();
            $data[ 'DATE_CREATION' ]    =   date_now();
            $data[ 'DATE_MOD' ]         =   date_now();
            return $data;
        });

        $crud->callback_before_update( function( $data ) {
            $data[ 'AUTHOR' ]           =   User::id();
            $data[ 'DATE_MOD' ]         =   date_now();
            return $data;
        });
        
        // XSS Cleaner
        $this->events->add_filter('grocery_callback_insert', array( $this->grocerycrudcleaner, 'xss_clean' ));
        $this->events->add_filter('grocery_callback_update', array( $this->grocerycrudcleaner, 'xss_clean' ));
        
        $crud->required_fields('NAME');

        $crud->change_field_type('DATE_CREATION', 'invisible');
        $crud->change_field_type('DATE_MOD', 'invisible');
        $crud->change_field_type('AUTHOR', 'invisible');

        $crud->set_relation('AUTHOR', 'aauth_users', 'name');
        $crud->set_relation('REF_UNIT', store_prefix() . 'gastro_materials_units', 'NAME' );
        $crud->set_relation('REF_MATERIAL', store_prefix() . 'gastro_materials', 'NAME' );

        $this->events->add_filter( 'grocery_header_buttons', function( $actions ) {
            $actions[]      =   [
                'text'      =>  __( 'New Adjustment', 'nexo' ),
                'url'       =>  site_url([ 'dashboard', store_slug(), 'raw-material', 'materialAdjustments', 'new' ])
            ];

            return $actions;
        });
        
        $crud->unset_jquery();
        $crud->unset_add();

        $output = $crud->render();
        
        foreach ($output->js_files as $files) {
            $this->enqueue->js(substr($files, 0, -3), '');
        }
        foreach ($output->css_files as $files) {
            $this->enqueue->css(substr($files, 0, -4), '');
        }
        
        return $output;
    }

    public function materialCrudHeader()
    {
        /**
		 * This feature is not more accessible on main site when
		 * multistore is enabled
		**/
		
		if( ( multistore_enabled() && ! is_multistore() ) ) {
			return show_error( __( 'This feature doesn\'t works on the warehouse.', 'raw-material' ) );
		}
        
        $crud = new Grocery_CRUD();

        $crud->set_theme('bootstrap');
        $crud->set_subject(__('Materials', 'raw-material' ));
        $crud->set_table( $this->db->dbprefix( store_prefix() . 'gastro_materials' ) );
		
		$fields					=	[
            'NAME',
            'SKU',
            'REF_UNIT_GROUP',
            'AUTHOR',
            'DESCRIPTION',
        ];
		$crud->columns( 'NAME', 'REF_UNIT_GROUP', 'SKU', 'AUTHOR', 'DATE_MOD' );
        $crud->fields( $fields );
        
        $state = $crud->getState();
        
        /**
         * Callback to support date formating
         * @since 3.12.8
         */
        $crud->callback_column( 'DATE_CREATION', function( $date ) {
            $datetime   =    new DateTime( $date ); 
            return $datetime->format( store_option( 'nexo_datetime_format', 'Y-m-d h:i:s' ) );
        });
        
        $crud->callback_column( 'DATE_MOD', function( $date ) {
            $datetime   =    new DateTime( $date ); 
            return $datetime->format( store_option( 'nexo_datetime_format', 'Y-m-d h:i:s' ) );
        });

        $crud->display_as( 'NAME',              __( 'Material Name', 'raw-material' ) );
        $crud->display_as( 'REF_UNIT_GROUP',    __( 'Unit Group Assigned', 'raw-material' ) );
        $crud->display_as( 'DESCRIPTION',       __( 'Descrpition', 'raw-material' ) );
        $crud->display_as( 'AUTHOR',            __( 'Author', 'raw-material' ) );
        $crud->display_as( 'DATE_CREATION',     __( 'Created On', 'raw-material' ) );
        $crud->display_as( 'DATE_MOD',          __( 'Modified On', 'raw-material' ) );

        $crud->callback_before_insert( function( $data ) {
            $data[ 'AUTHOR' ]           =   User::id();
            $data[ 'DATE_CREATION' ]    =   date_now();
            $data[ 'DATE_MOD' ]         =   date_now();
            return $data;
        });

        $crud->callback_before_update( function( $data ) {
            $data[ 'AUTHOR' ]           =   User::id();
            $data[ 'DATE_MOD' ]         =   date_now();
            return $data;
        });

        $crud->add_action(
			__( 'Unit Quantities', 'raw-material'),
            '',
            '',
			'fa fa-list',
			[ $this, 'getLink' ]
        );
        
        // XSS Cleaner
        $this->events->add_filter('grocery_callback_insert', array( $this->grocerycrudcleaner, 'xss_clean' ));
        $this->events->add_filter('grocery_callback_update', array( $this->grocerycrudcleaner, 'xss_clean' ));
        
        $crud->required_fields('NAME');

        $crud->change_field_type('DATE_CREATION', 'invisible');
        $crud->change_field_type('DATE_MOD', 'invisible');
        $crud->change_field_type('AUTHOR', 'invisible');

        $crud->set_relation('AUTHOR', 'aauth_users', 'name');
        $crud->set_relation('REF_UNIT_GROUP', store_prefix() . 'gastro_materials_units_groups', 'NAME' );
        
        $crud->unset_jquery();
        $output = $crud->render();
        
        foreach ($output->js_files as $files) {
            $this->enqueue->js(substr($files, 0, -3), '');
        }
        foreach ($output->css_files as $files) {
            $this->enqueue->css(substr($files, 0, -4), '');
        }
        
        return $output;
    }

    public function getLink( $id )
    {
        return site_url([ 'dashboard', store_slug(), 'raw-material', 'materials', 'quantities', $id ]);
    }

    public function materialUnit( $material_id, $unit_id, $page = 1 )
    {
        $perPage                    =   20;
        $materialService            =   new Material;
        $history                    =   $materialService->getMaterialUnitHistory( $material_id, $unit_id, $perPage );
        $history[ 'entries' ]       =   collect( $history[ 'entries' ] )->map( function( $entry ) use ( $materialService ){
            $entry->recipe          =   $materialService->getRecipe( $entry->REF_RECIPE );
            return $entry;
        })->toArray();
        $material                   =   $materialService->getMaterial( $material_id );
        $unit                       =   $materialService->getUnit( $unit_id );
        $users                      =   $this->auth->list_users();
        $dom                        =   $this->load->module_view( 'raw-material', 'materials.material-unit', compact( 'history', 'material', 'unit', 'users' ), true );

        $this->Gui->set_title( sprintf( __( 'Material Unit Summary : &mdash; %s', 'raw-material' ), $material[ 'NAME' ] ) );
        $this->load->module_view( 'raw-material', 'common.dom', compact( 'dom' ) );
    }
}