<?php
namespace RawMaterial\Classes;

use User;
use Carbon\Carbon;
use Tendoo_Module;
use RawMaterial\Exceptions\NotFoundRecipeException;
use RawMaterial\Exceptions\NotEnoughMaterialException;

class Material extends Tendoo_Module
{
    public function unitGroupHasDefault( $id )
    {
        $units  =   $this->db->where( 'REF_UNIT_GROUP', $id )
            ->where( 'BASE_UNIT', '1' )
            ->get( store_prefix() . 'gastro_materials_units' )
            ->result_array();
        return $units;
    }

    /**
     * Define a specific unit as the
     * default id.
     * @param int unit id
     * @return mixed
     */
    public function switchBaseUnit( $unit_id )
    {
        $unit       =   $this->getUnit( $unit_id );

        if ( ! $unit ) {
            return $unit;
        }

        $units      =   $this->getUnitsFromGroup( $unit[ 'REF_UNIT_GROUP' ] );

        foreach( $units as $unit ) {
            if ( ( int ) $unit[ 'ID' ] !== ( int ) $unit_id ) {
                $this->db->where( 'ID', $unit[ 'ID' ])
                    ->update( store_prefix() . 'gastro_materials_units', [
                        'BASE_UNIT' =>  0
                    ]);
            } else {
                $this->db->where( 'ID', $unit[ 'ID' ] )
                    ->update( store_prefix() . 'gastro_materials_units', [
                        'BASE_UNIT' =>  1
                    ]);
            }
        }        
    }

    /**
     * Update a provided ingredient
     * @param int ingredient id
     * @param array values
     * @return void
     */
    public function updateIngredient( $id, $ingredient )
    {
        $this->db->where( 'ID', $id )
            ->update( store_prefix() . 'gastro_materials_recipes_items', [
                'REF_MATERIAL'  =>  $ingredient[ 'material' ],
                'QUANTITY'      =>  $ingredient[ 'quantity' ],
                'REF_UNIT'      =>  $ingredient[ 'uom' ],
                'AUTHOR'        =>  User::id(),
                'DATE_CREATION' =>  date_now(),
                'DATE_MOD'      =>  date_now(),
            ]);
    }

    public function addIngredient( $recipe_id, $ingredient )
    {
        $this->db->insert( store_prefix() . 'gastro_materials_recipes_items', [
            'REF_RECIPE'    =>  $recipe_id,
            'REF_MATERIAL'  =>  $ingredient[ 'material' ],
            'QUANTITY'      =>  $ingredient[ 'quantity' ],
            'REF_UNIT'      =>  $ingredient[ 'uom' ],
            'AUTHOR'        =>  User::id(),
            'DATE_CREATION' =>  date_now(),
            'DATE_MOD'      =>  date_now(),
        ]);
    }

    public function removeIngredient( $ingredient_id )
    {
        $this->db->where( 'ID', $ingredient_id )
            ->delete( store_prefix() . 'gastro_materials_recipes_items' );
    }

    /**
     * get a single unit
     * @param int unit id
     * @return mixed
     */
    public function getUnit( $unit_id )
    {
        $unit   =   $this->db->where( 'ID', $unit_id )
            ->get( store_prefix() . 'gastro_materials_units' )
            ->result_array();

        if ( ! empty( $unit ) ) {
            return $unit[0];
        }

        return false;
    }
    
    /**
     * get all units that belongs to a specific group
     * @param int group id
     * @return array
     */
    public function getUnitsFromGroup( $group_id )
    {
        return $this->db->where( 'REF_UNIT_GROUP', $group_id )
            ->get( store_prefix() . 'gastro_materials_units' )
            ->result_array();
    }

    /**
     * Return a list of available material
     * @return array Materials
     */
    public function getMaterials()
    {
        return $this->db->get( store_prefix() . 'gastro_materials' )
            ->result_array();
    }

    /**
     * get units groups
     * @return array
     */
    public function getUnitGroupWithEntities()
    {
        $groups  =   $this->db->get( store_prefix() . 'gastro_materials_units_groups' )
            ->result_array();
        foreach( $groups as &$group ) {
            $group[ 'units' ]   =   $this->getUnitsFromGroup( $group[ 'ID' ] );
        }

        return $groups;
    }

    /**
     * get single unit group
     * with entities
     * @param integer group id
     * @return array
     */
    public function getUnitGroup( $id )
    {
        $groups  =   $this->db->where( 'ID', $id )->get( store_prefix() . 'gastro_materials_units_groups' )
            ->result_array();

        if ( $groups ) {
            foreach( $groups as &$group ) {
                $group[ 'units' ]   =   $this->getUnitsFromGroup( $group[ 'ID' ] );
            }
    
            return $groups[0];
        }

        return false;
    }

    /**
     * Save a recipe with the included 
     * ingredient to the database
     * @param string name
     * @param array ingredients
     * @param string description
     * @param mixed id
     */
    public function saveRecipe( $name, $ingredients, $description = null, $id = null )
    {
        $this->db->insert( store_prefix() . 'gastro_materials_recipes', [
            'NAME'          =>  $name,
            'AUTHOR'        =>  User::id(),
            'DATE_CREATION' =>  date_now(),
            'DATE_MOD'      =>  date_now(),
            'DESCRIPTION'   =>  $description,
        ]);

        $refid          =   $this->db->insert_id();

        foreach( $ingredients as $ingredient ) {
            $this->db->insert( store_prefix() . 'gastro_materials_recipes_items', [
                'REF_RECIPE'    =>  $refid,
                'REF_MATERIAL'  =>  $ingredient[ 'material' ],
                'QUANTITY'      =>  $ingredient[ 'quantity' ],
                'REF_UNIT'      =>  $ingredient[ 'uom' ],
                'AUTHOR'        =>  User::id(),
                'DATE_CREATION' =>  date_now(),
                'DATE_MOD'      =>  date_now(),
            ]);
        }

        nexo_log(
            __( 'Material Recipe', 'raw-material' ),
            sprintf(
                __( 'The user %s has created a raw material recipe <strong>%s</strong>.', 'raw-material' ),
                User::pseudo(),
                $name
            )
        );

        return [
            'status'    =>  'success',
            'message'   =>  __( 'The recipe has been created', 'raw-material' )
        ];
    }

    /**
     * @param int $id
     * @param string $name
     * @param string $description
     */
    public function updateRecipe( $id, $name, $description = null )
    {
        $this->db->where( 'id', $id )->update( store_prefix() . 'gastro_materials_recipes', [
            'NAME'          =>  $name,
            'AUTHOR'        =>  User::id(),
            'DATE_CREATION' =>  date_now(),
            'DATE_MOD'      =>  date_now(),
            'DESCRIPTION'   =>  $description,
        ]);
    }
    
    public function getMaterialsUsingGroup( $group_id )
    {
        return $this->db->where( 'REF_UNIT_GROUP', $group_id )
            ->get( store_prefix() . 'gastro_materials' )
            ->result_array();
    }

    /**
     * Determine wether a unit group
     * is used or not
     * @param int group id
     * @return boolean
     */
    public function isUnitGroupUsed( $group_id )
    {
        return  $this->db->where( 'REF_UNIT_GROUP', $group_id )
            ->get( store_prefix() . 'gastro_materials' )
            ->num_rows() > 0;        
    }
    
    /**
     * Delete all the units that has been
     * attached to a specific unit group
     * @param int unit group id
     * @return void
     */
    public function deleteUnitGroupEntities( $group_id )
    {
        $this->db->where( 'REF_UNIT_GROUP', $group_id )
            ->delete( store_prefix() . 'gastro_materials_units' );
    }

    /**
     * Delete Recipe ingredients
     * @param int recipe id
     * @return void
     */
    public function deleteRecipeIngredients( $recipe_id )
    {
        $this->db->where( 'REF_RECIPE', $recipe_id )
            ->delete( store_prefix() . 'gastro_materials_recipes_items' );
    }

    public function getRecipe( $id )
    {
        $recipe     =   $this->db->where( 'ID', $id )
            ->get( store_prefix() . 'gastro_materials_recipes' )
            ->result_array();

        if ( $recipe ) {
            $recipe                     =   $recipe[0];
            $recipe[ 'ingredients' ]    =   $this->db->where( 'REF_RECIPE', $id )
                ->get( store_prefix() . 'gastro_materials_recipes_items' )
                ->result_array();
            return $recipe;            
        }

        return false;
    }

    public function getRecipes()
    {
        return $this->db
            ->get( store_prefix() . 'gastro_materials_recipes' )
            ->result_array();
    }

    public function getUnits()
    {
        return $this->db->get( store_prefix() . 'gastro_materials_units' )
            ->result_array();
    }

    /**
     * create a procurement and attach
     * items to it
     * @param string title
     * @param array items
     * @return array result
     */
    public function createAndSupplyProcurement( $title, $items )
    {
        $this->db->insert( store_prefix() . 'gastro_materials_supplies', [
            'NAME'          =>      $title,
            'TOTAL_COST'    =>      0,
            'TOTAL_ITEMS'   =>      0,
            'AUTHOR'        =>      User::id(),
            'DATE_CREATION' =>      date_now(),
            'DATE_MOD'      =>      date_now()
        ]);

        $procurement_id     =   $this->db->insert_id();

        nexo_log(
            __( 'Material Supply', 'raw-material' ),
            sprintf(
                __( 'The user %s has made a material supply named <strong>%s</strong>.', 'raw-material' ),
                User::pseudo(),
                $title
            )
        );

        return $this->__addItemsToProcurement( $procurement_id, $items );
    }

    private function __addItemsToProcurement( $procurement_id, $items )
    {
        foreach( $items as $item ) {
            $this->db->insert( store_prefix() . 'gastro_materials_supplies_items', [
                'REF_SUPPLY'        =>  $procurement_id,
                'REF_UNIT'          =>  $item[ 'uom' ],
                'REF_MATERIAL'      =>  $item[ 'material' ],
                'QUANTITY'          =>  $item[ 'quantity' ],
                'AUTHOR'            =>  User::id(),
                'DATE_CREATION'     =>  date_now(),
                'DATE_MOD'          =>  date_now(),
                'PURCHASE_PRICE'    =>  $item[ 'purchase_price' ],
                'TOTAL_PRICE'       =>  floatval( $item[ 'purchase_price' ]) * floatval( $item[ 'quantity' ] ),
            ]);

            $this->increaseStockUnit( 
                $item[ 'material' ],
                $item[ 'uom' ],
                $item[ 'quantity' ]
            );

            $this->recordIngredientTracking( 'procurement', $item[ 'material' ], $item[ 'uom' ], $item[ 'quantity' ] );
        }

        $this->refreshProcurement( $procurement_id );

        return [
            'status'    =>  'success',
            'message'   =>  __( 'The procurement has been created.', 'raw-material' )
        ];
    }

    /**
     * Refresh a specific procurement
     * @param int procurement id
     * @return void
     */
    public function refreshProcurement( $id )
    {
        $items              =   $this->getProcurementItems( $id );
        $totalItems         =   count( $items );
        $totalCost          =   0;

        foreach( $items as $item ) {
            $totalCost      +=   ( floatval( $item[ 'QUANTITY' ] ) * floatval( $item[ 'PURCHASE_PRICE' ] ) );
        }

        $this->db->where( 'ID', $id )
            ->update( store_prefix() . 'gastro_materials_supplies', [
                'DATE_MOD'      =>  date_now(),
                'AUTHOR'        =>  User::id(),
                'TOTAL_COST'    =>  $totalCost,
                'TOTAl_ITEMS'   =>  $totalItems
            ]);
    }

    /**
     * return a procurements items
     * @param int procurement id
     * @return array
     */
    public function getProcurementItems( $id )
    {
        return $this->db->where( 'REF_SUPPLY', $id )
            ->get( store_prefix() . 'gastro_materials_supplies_items' )
            ->result_array();
    }

    public function increaseStockUnit( int $material_id, int $unit_id, float $quantity )
    {
        $previousStock       =   $this->getMaterialUnitStock( $material_id, $unit_id );

        if ( empty( $previousStock ) ) {
            $this->setMaterialUnitStock( $material_id, $unit_id, $quantity );
        } else {
            $this->setMaterialUnitStock( $material_id, $unit_id, floatval( $previousStock[ 'QUANTITY'] ) + $quantity );
        }

        return [
            'status'    =>  'success',
            'message'   =>  __( 'The material unit stock has been updated', 'raw-material' )
        ];
    }

    /**
     * Get a specific material entry
     * @param int material id
     * @return mixed array or null
     */
    public function getMaterial( $material_id )
    {
        $material   =   $this->db->where( 'ID', $material_id )
            ->get( store_prefix() . 'gastro_materials' )
            ->result_array();

        return ! empty( $material ) ? $material[0] : null;
    }

    /**
     * Get a specific material using SKU
     * @param string material SKU
     * @return mixed array or null
     */
    public function getMaterialUsingSKU( $sku )
    {
        $material   =   $this->db->where( 'SKU', $sku )
            ->get( store_prefix() . 'gastro_materials' )
            ->result_array();

        return ! empty( $material ) ? $material[0] : null;
    }

    public function decreaseStockUnit( $material_id, $unit_id, $quantity )
    {
        $previousStock          =   $this->getMaterialUnitStock( $material_id, $unit_id );
        $material               =   $this->getMaterial( $material_id );

        if ( empty( $previousStock ) ) {
            $message        =   __( 'Unable to decrease stock for a unit that has\'nt been procured', 'raw-material' );
            log_message( 'error', $message );

            return [
                'status'    =>  'failed',
                'message'   =>  $message
            ];
        } else {
            
            if ( floatval( $previousStock[ 'QUANTITY' ] ) - $quantity >= 0 ) {
                $this->db->where( 'ID', $previousStock[ 'ID' ] )
                    ->update( store_prefix() . 'gastro_materials_units_stock', [
                        'QUANTITY'  =>  floatval( $previousStock[ 'QUANTITY'] ) - $quantity,
                    ]);

                return [
                    'status'    =>  'success',
                    'message'   =>  __( 'The material unit stock has been updated', 'raw-material' )
                ];
            } else {
                $message        =   __( 'Unable to reduce the stock for the material %s with the unit provided', 'raw-material' );

                return [
                    'status'    =>  'failed',
                    'message'   =>  $message,
                ];
            }
        }
    }

    public function getMaterialUnitStock( int $material_id, int $unit_id )
    {
        $unitStock  =   $this->db->where( 'REF_MATERIAL', $material_id )
            ->where( 'REF_UNIT', $unit_id )
            ->get( store_prefix() . 'gastro_materials_units_stock' )
            ->result_array();

        return ! empty( $unitStock ) ? $unitStock[0] : [];
    }

    public function getSupplies()
    {
        return $this->db->get( store_prefix() . 'gastro_materials_supplies' )
            ->result_array();
    }

    public function addMaterialUnitToProcurement( $id, $items )
    {
        return $this->__addItemsToProcurement( $id, $items );
    }

    public function getMaterialStock( int $material_id )
    {
        return   $this->db
            ->where( 'REF_MATERIAL', $material_id )
            ->get( store_prefix() . 'gastro_materials_units_stock' )
            ->result_array();
    }

    public function getMaterialStocks()
    {
        return   $this->db
            ->get( store_prefix() . 'gastro_materials_units_stock' )
            ->result_array();
    }

    public function getGroupBaseUnit( $group_id )
    {
        $units      =   $this->db->where( 'REF_UNIT_GROUP', $group_id )
            ->where( 'BASE_UNIT', 1 )
            ->get( store_prefix() . 'gastro_materials_units' )
            ->result_array();

        return ! empty( $units ) ? $units[0] : false;
    }

    public function getGroups()
    {
        return $this->db
            ->get( store_prefix() . 'gastro_materials_units_groups' )
            ->result_array();
    }

    public function getSupply( $id )
    {
        $procurement    =   $this->db->where( 'ID', $id )
            ->get( store_prefix() . 'gastro_materials_supplies' )
            ->result_array();

        if ( $procurement ) {
            $procurement                =   $procurement[0];
            $procurement[ 'items' ]   =   $this->getProcurementItems( $id );
            return $procurement;
        }

        return false;
    }

    /**
     * Consume material from a recipe
     * @param int recipe id
     * @param float times or quantity
     * @return void
     */
    public function depleteMaterialsFromRecipe( $recipe_id, $times = 1, $product_id = 0, $order_product_id = 0 )
    {
        $recipe     =   $this->getRecipe( $recipe_id );
        $item       =   get_instance()->item_model->getUsingID( $product_id );

        if ( ! empty( $recipe ) ) {
            $ingredients    =   $recipe[ 'ingredients' ];
            $flyStock       =   [];
            $endResult      =   [];

            foreach( $ingredients as $ingredient ) {
                $currentUnitStock       =   $this->getMaterialUnitStock( $ingredient[ 'REF_MATERIAL' ], $ingredient[ 'REF_UNIT' ] );
                $unit                   =   $this->getUnit( $ingredient[ 'REF_UNIT' ] );
                $quantity               =   floatval( $currentUnitStock[ 'QUANTITY' ] ) - ( floatval( $ingredient[ 'QUANTITY' ] ) * $times );
                
                $this->setMaterialUnitStock( $ingredient[ 'REF_MATERIAL' ], $ingredient[ 'REF_UNIT' ], $quantity );
                $this->recordIngredientTracking( 
                    'consumption',
                    $ingredient[ 'REF_MATERIAL' ], 
                    $ingredient[ 'REF_UNIT' ], 
                    floatval( $ingredient[ 'QUANTITY' ] ) * $times, 
                    $recipe_id, 
                    $product_id,
                    $order_product_id
                );
            }

            $this->recordRecipeTracking( $recipe_id, $times, 'consumption', $product_id, $order_product_id );

            $model      =   get_instance()->load->module_model( 'nexo', 'NexoItems', 'item_model' );

            nexo_log(
                __( 'Recipe Sold', 'raw-material' ),
                sprintf(
                    __( 'The user <strong>%s</strong> has sold the item <strong>%s</strong> made with the recipe <strong>%s</strong>.', 'raw-material' ),
                    User::pseudo(),
                    @$item[ 'DESIGN' ] ?: __( 'N/A', 'raw-material' ),
                    $recipe[ 'NAME' ]
                )
            );
        } else {
            nexo_log( 
                __( 'Recipe Not Found', 'raw-material' ),
                sprintf(
                    __( 'The product <strong>"%s"</strong> has been sold by "%s", but no valid recipe has been assigned to that. The material can\'t then be consummed.', 'raw-material' ),
                    @$item[ 'DESIGN' ] ?: __( 'N/A', 'raw-material' ),
                    User::pseudo()
                )
            );
        }
    }

    public function recordRecipeTracking( $recipe_id, $times, $type, $product_id, $order_product_id )
    {
        $this->db->insert( store_prefix() . 'gastro_materials_recipes_summary', [
            'REF_RECIPE'            =>  $recipe_id,
            'REF_PRODUCT'           =>  $product_id,
            'REF_ORDER_PRODUCT'     =>  $order_product_id,
            'QUANTITY'              =>  $times,
            'AUTHOR'                =>  User::id(),
            'DATE_CREATION'         =>  date_now(),
            'STATUS'                =>  $type
        ]);
    }

    /**
     * Record an ingredient consumption. This reduce the ingredient
     * stock and record a consumption history
     * @param int material id
     * @param int unit id
     * @param float quantity
     * @param float old quantity
     * @param int product id
     * @return array
     */
    public function recordIngredientTracking( string $type, int $material_id, int $unit_id, float $quantity, int $recipe_id = 0, int $product_id = 0, int $order_product_id = 0 )
    {
        $this->db->insert( store_prefix() . 'gastro_materials_usage_summary', [
            'STATUS'            =>  $type,
            'REF_PRODUCT'       =>  $product_id,
            'REF_ORDER_PRODUCT' =>  $order_product_id,
            'REF_UNIT'          =>  $unit_id,
            'REF_MATERIAL'      =>  $material_id,
            'REF_RECIPE'        =>  $recipe_id,
            'QUANTITY'          =>  $quantity,
            'AUTHOR'            =>  User::id(),
            'DATE_CREATION'     =>  date_now(),
            'DATE_MOD'          =>  date_now()
        ]);
    }

    /**
     * Defines a material stock quantity
     * if the material record doesn't exist, 
     * it's created
     * @param int material id
     * @param int unit id
     * @param float quantity
     * @return void
     */
    public function setMaterialUnitStock( $material_id, $unit_id, $quantity )
    {
        $existing   =   $this->db->where( 'REF_MATERIAL', $material_id )
            ->where( 'REF_UNIT', $unit_id )
            ->get( store_prefix() . 'gastro_materials_units_stock' )
            ->result_array();
        
        if ( ! empty( $existing ) ) {
            $this->db->where( 'REF_MATERIAL', $material_id )
                ->where( 'REF_UNIT', $unit_id )
                ->update( store_prefix() . 'gastro_materials_units_stock', [
                    'QUANTITY'  =>  $quantity
                ]);
        } else {
            $this->db->insert( store_prefix() . 'gastro_materials_units_stock', [
                'QUANTITY'      =>  $quantity,
                'REF_MATERIAL'  =>  $material_id,
                'REF_UNIT'      =>  $unit_id,
                'AUTHOR'        =>  User::id(),
                'DATE_CREATION' =>  date_now(),
            ]);
        }

        return [
            'status'    =>  'success',
            'message'   =>  __( 'The material stock has been updated', 'raw-material' )
        ];
    }

    public function canConsumeRecipe( $recipe_id, $times = 1, $flyStock = [] )
    {
        $recipe             =   $this->getRecipe( $recipe_id );

        if ( ! empty( $recipe ) ) {
            $ingredients    =   $recipe[ 'ingredients' ];
            $endResult      =   [];

            foreach( $ingredients as $ingredient ) {
                $result         =   $this->fakelyConsumeIngredient( $ingredient, $flyStock, $times );
                $flyStock       =   $result[ 'data' ][ 'flyStock' ];
                $endResult[]    =   $result;
            }

            $hasFailure         =   collect( $endResult )->filter( function( $operation ) {
                return $operation[ 'status' ] === 'failed';
            })->count() > 0;

            if ( $hasFailure ) {
                return [
                    'status'    =>  'failed',
                    'message'   =>  __( 'Unable to proceed as one of the ingredient is not enough', 'raw-material' ),
                    'data'      =>  [
                        'result'    =>  $endResult,
                        'flyStock'  =>  $flyStock
                    ],
                    'class'     =>  NotEnoughMaterialException::class
                ];
            } else {
                return [
                    'status'    =>  'success',
                    'message'   =>  __( 'The material can be consummed.', 'raw-material' ),
                    'data'      =>  [
                        'result'    =>  $endResult,
                        'flyStock'  =>  $flyStock
                    ]
                ];
            }
        }

        return [
            'status'    =>  'failed',
            'message'   =>  __( 'Unable to retreive the recipe you\'re looking for', 'raw-material' ),
            'class'     =>  NotFoundRecipeException::class,
            'data'      =>  [
                'flyStock'  =>  $flyStock
            ]
        ];
    }

    public function fakelyConsumeIngredient( array $ingredient, array $flyStock, int $times )
    {
        $currentUnitStock    =   collect( $flyStock )->filter( function( $stock, $key ) use ( $ingredient ) {
            return $ingredient[ 'REF_MATERIAL' ] . '-' . $ingredient[ 'REF_UNIT' ] === $key;
        })->first();

        $wasFound           =   true;

        if ( empty( $currentUnitStock ) ) {
            $wasFound           =   false;
            $currentUnitStock   =   $this->getMaterialUnitStock( $ingredient[ 'REF_MATERIAL' ], $ingredient[ 'REF_UNIT' ] );

            if( empty( $currentUnitStock ) ) {
                
                $material   =   $this->getMaterial( $ingredient[ 'REF_MATERIAL' ] );
                $unit       =   $this->getUnit( $ingredient[ 'REF_UNIT' ] );

                return [
                    'status'    =>  'failed',
                    'message'   =>  sprintf(
                        __( 'No stock exists for the material %s using the unit %s', 'raw-material' ),
                        $material[ 'NAME' ],
                        $unit[ 'NAME' ]
                    ),
                    'data'      =>  compact( 'ingredient', 'flyStock', 'unit', 'material', 'currentUnitStock' )
                ];
            }
        }

        $unit                   =   $this->getUnit( $ingredient[ 'REF_UNIT' ] );
        $operationResult        =   ( floatval( $unit[ 'UNIT_VALUE' ] ) * ( $times * floatval( $ingredient[ 'QUANTITY' ] ) ) );


        $afterDeduction         =   ( floatval( $currentUnitStock[ 'QUANTITY' ] ) * floatval( $unit[ 'UNIT_VALUE' ] ) ) - $operationResult;

        if ( $afterDeduction >= 0 ) {
            if ( $wasFound ) {

                /**
                 * if the related stock has been found, 
                 * we deplete that stock and save the flyStock
                 */
                $flyStock   =   collect( $flyStock )->mapWithKeys( function( $stock ) use ( $ingredient, $afterDeduction ) {
                    if ( $stock[ 'REF_UNIT' ] === $ingredient[ 'REF_UNIT' ] && $stock[ 'REF_MATERIAL' ] === $ingredient[ 'REF_MATERIAL' ] ) {
                        $stock[ 'QUANTITY' ]    =   $afterDeduction;
                    }
                    return [ ($stock[ 'REF_MATERIAL' ] . '-' . $stock[ 'REF_UNIT' ]) => $stock ];
                })->toArray();


            } else {

                /**
                 * if the flyStock has not been found
                 * we'll create a new flyStock entry and save 
                 * the values there
                 */
                $stock                  =   $currentUnitStock;
                $stock[ 'QUANTITY' ]    =   $afterDeduction;
                $flyStock               =   collect( $flyStock )->put( 
                    $stock[ 'REF_MATERIAL' ] . '-' . $stock[ 'REF_UNIT' ], 
                    $stock 
                )->toArray();
            }

            return [
                'status'    =>  'success',
                'message'   =>  __( 'The material can be consummed', 'raw-material' ),
                'data'      =>  compact( 'ingredient', 'flyStock', 'afterDeduction', 'unit', 'operationResult', 'currentUnitStock' )
            ];
        }

        $base           =   $this->getGroupBaseUnit( $unit[ 'REF_UNIT_GROUP' ] );
        $siblings       =   $this->events->apply_filters( 
            'raw_material_sibling_unit_stock', 
            false,
            $ingredient, 
            $unit,
            $base
        );

        /**
         * If fetching the siblings hasn't been
         * overrided by any other module, let's proceed.
         */
        if ( $siblings === false ) {
            $siblings   =   $this->getSiblingUnitsStock( $ingredient[ 'REF_MATERIAL' ], $ingredient[ 'REF_UNIT' ] );
        }

        /**
         * We'll try to determine
         * if after a automatic conversion
         * the stock will be fullfilled.
         */
        if ( ! empty( $siblings ) && ( bool ) $currentUnitStock[ 'INCOMING_CONVERSION' ] ) {
            $hasFoundMatch  =   false;
            $convertible    =   collect( $siblings )->map( function( $unitStock ) use ( $base, $operationResult, &$flyStock, &$hasFoundMatch, $unit, $ingredient ) {
                if ( ! $hasFoundMatch ) {
                    /**
                     * get the unitStock from the flyStock if it's defined
                     * otherwise, uses what has been provided by the siblings.
                     */
                    $unitStock    =   collect( $flyStock )->filter( function( $stock, $key ) use ( $unitStock ) {
                        return $unitStock[ 'REF_MATERIAL' ] . '-' . $unitStock[ 'REF_UNIT' ] === $key;
                    })->first() ?? $unitStock;
    
                    if ( ( bool ) $unitStock[ 'OUTGOING_CONVERSION' ] && floatval( $unitStock[ 'QUANTITY' ] ) > 0 ) {
                        $unitOriginal               =   $this->getUnit( $unitStock[ 'REF_UNIT' ] );
                        $material                   =   $this->getMaterial( $unitStock[ 'REF_MATERIAL' ] );
                        $totalAvailable             =   floatval( $unitStock[ 'QUANTITY' ] ) * floatval( $unitOriginal[ 'UNIT_VALUE' ] );
                        $totalToConvert             =   ceil( $operationResult / floatval( $unitOriginal[ 'UNIT_VALUE' ] ) );
                        $afterDeduction             =   floatval( $unitStock[ 'QUANTITY' ] ) - ceil( $totalToConvert );
                        $key                        =   $unitStock[ 'REF_MATERIAL' ] . '-' . $unitStock[ 'REF_UNIT' ];

                        /**
                         * We're updating the Unit Stock 
                         * with the convertable unit. This means,
                         * reduce from the original unit and updating
                         * the destination unit on the virtual storage.
                         */
                        if ( $afterDeduction >= 0 ) {
                            if ( in_array( $key, collect( $flyStock )->keys()->toArray() ) ) {
                                
                                $flyStock   =   collect( $flyStock )->mapWithKeys( function( $stock ) use ( $unitStock, $afterDeduction ) {
                                    if ( $stock[ 'REF_UNIT' ] === $unitStock[ 'REF_UNIT' ] && $stock[ 'REF_MATERIAL' ] === $unitStock[ 'REF_MATERIAL' ] ) {
                                        $stock[ 'QUANTITY' ]    =   $afterDeduction;
                                    }
                                    return [ ($stock[ 'REF_MATERIAL' ] . '-' . $stock[ 'REF_UNIT' ]) => $stock ];
                                })->toArray();

                            } else {
                                $unitStock[ 'QUANTITY' ]    =   $afterDeduction;
                                $flyStock   =   collect( $flyStock )->put( $key, $unitStock )->toArray();
                            }

                            /**
                             * let's populate the converted stock
                             * so that it can be used.
                             */
                            $toKey              =   $ingredient[ 'REF_MATERIAL' ] . '-' . $ingredient[ 'REF_UNIT' ];

                            /**
                             * For the fake stock available. We'll remove the 
                             * requested quantity so that over the consumption
                             * an exact number of quantity remains.
                             */
                            $totalConverted     =   floor( 
                                ( floatval( $unitOriginal[ 'UNIT_VALUE' ] ) * $totalToConvert ) / floatval( $unit[ 'UNIT_VALUE' ] ) 
                            ) - $operationResult;

                            if ( in_array( $toKey, collect( $flyStock )->keys()->toArray() ) ) {
                                $flyStock   =   collect( $flyStock )->mapWithKeys( function( $stock ) use ( $ingredient, $totalToConvert, $totalConverted ) {
                                    if ( $stock[ 'REF_UNIT' ] === $ingredient[ 'REF_UNIT' ] && $stock[ 'REF_MATERIAL' ] === $ingredient[ 'REF_MATERIAL' ] ) {
                                        $stock[ 'QUANTITY' ]    =   intval( $stock[ 'QUANTITY' ] ) + $totalConverted;
                                    }
                                    return [ ($stock[ 'REF_MATERIAL' ] . '-' . $stock[ 'REF_UNIT' ]) => $stock ];
                                })->toArray();
                            } else {
                                $_currentStock  =   $this->getMaterialUnitStock( 
                                    $ingredient[ 'REF_MATERIAL' ], 
                                    $ingredient[ 'REF_UNIT' ]
                                );

                                $_currentStock[ 'QUANTITY' ]    =   floatval( $_currentStock[ 'QUANTITY' ] ) + $totalConverted;

                                $flyStock   =   collect( $flyStock )->put( 
                                    $ingredient[ 'REF_MATERIAL' ] . '-' . $ingredient[ 'REF_UNIT' ],
                                    $_currentStock
                                )->toArray();
                            }
                            
                            $hasFoundMatch  =   true;

                            return [
                                'status'    =>  'success',
                                'message'   =>  __( 'A conversion can be successfully made.', 'raw-material' ),
                                'data'      =>  [
                                    'fromUnit'          =>  $unitOriginal,
                                    'toUnit'            =>  $unit,
                                    'convert'           =>  ceil( $totalToConvert ),
                                    'afterDeduction'    =>  $afterDeduction,
                                    'totalAvailable'    =>  $totalAvailable,
                                    'totalToConvert'    =>  $totalToConvert,
                                    'flyStock'          =>  $flyStock,
                                    'ingredient'        =>  $ingredient,
                                    'material'          =>  $material,
                                    'unit'              =>  $unit
                                ]
                            ];
                        }

                        return [
                            'status'    =>  'failed',
                            'message'   =>  __( 'The conversion will cause negative stock, so it can\'t be made.', 'raw-material' ),
                            'data'      =>  [
                                'afterDeduction'    =>  $afterDeduction,
                                'totalAvailable'    =>  $totalAvailable,
                                'totalToConvert'    =>  $totalToConvert,
                                'flyStock'          =>  $flyStock,
                                'ingredient'        =>  $ingredient,
                                'material'          =>  $material,
                                'unit'              =>  $unit
                            ]
                        ];
                    }
                }
                return false;
            })->reject( function( $stock ) {
                return $stock === false;
            })->first();
        }

        $material       =   $this->getMaterial( $ingredient[ 'REF_MATERIAL' ]);

        if ( isset( $convertible ) ) {
            return $convertible;
        }

        return [
            'status'    =>  'failed',
            'message'   =>  sprintf( __( '%s can\'t be consummed. The stock is not enough.', 'raw-material' ), $material[ 'NAME' ] ),
            'data'      =>  compact( 'ingredient', 'flyStock', 'afterDeduction', 'unit', 'operationResult', 'currentUnitStock', 'material' )
        ];
    }

    /**
     * Get material unit history paginated
     * @param int material id
     * @param int unit id
     * @param int per page
     * @return array
     */
    public function getMaterialUnitHistory( $material_id, $unit_id, $perPage = 10 )
    {
        return pagination([
            'table'         =>  'gastro_materials_usage_summary',
            'unit_id'       =>  $unit_id,
            'material_id'   =>  $material_id,
        ]);
    }


    /**
     * Delete a procurement items
     * @param int procurement id
     * @return array
     */
    public function deleteProcurementEntries( $procurement_id )
    {
        $items  =   $this->getProcurementItems( $procurement_id );

        collect( $items )->each( function( $item ) {
            $stock      =   $this->getMaterialUnitStock( $item[ 'REF_MATERIAL' ], $item[ 'REF_UNIT' ] );
            $remaining  =   floatval( $stock[ 'QUANTITY' ] ) - floatval( $item[ 'QUANTITY' ] );
            
            /**
             * we would like to ge the proper
             * remaining value counting what has been sold
             */
            if ( $remaining < 0 ) {
                $removed    =   collect( range( 0, floatval( $item[ 'QUANTITY' ] ) ) )->reduce( function( $before, $after ) {
                    return floatval( $before ) + floatval( $after );
                });
            } else {
                $removed    =   $remaining;
            }

            $remaining  =   $remaining < 0 ? 0 : $remaining;
            

            $this->setMaterialUnitStock( $item[ 'REF_MATERIAL' ], $item[ 'REF_UNIT' ], $remaining );
            $this->recordIngredientTracking( 'deletion', $item[ 'REF_MATERIAL' ], $item[ 'REF_UNIT' ], $removed, floatval( $stock[ 'QUANTITY' ] ) );
            
            $this->db->where( 'ID', $item[ 'ID' ])
                ->delete( store_prefix() . 'gastro_materials_supplies_items' );
        });

        return [
            'status'    =>  'success',
            'message'   =>  __( 'The entries has been deleted.', 'raw-material' )
        ];
    }

    public function restoreMaterialFromRecipeSilently( int $recipe_id, int $order_product_id, float $times )
    {
        $recipe     =   $this->getRecipe( $recipe_id );

        if ( $recipe ) {
            foreach( $recipe[ 'ingredients' ] as $ingredient ) {
                $currentUnitStock       =   $this->getMaterialUnitStock( $ingredient[ 'REF_MATERIAL' ], $ingredient[ 'REF_UNIT' ] );
                $unit                   =   $this->getUnit( $ingredient[ 'REF_UNIT' ] );
                $quantity               =   floatval( $currentUnitStock[ 'QUANTITY' ] ) + ( floatval( $ingredient[ 'QUANTITY' ] ) * $times );
                
                $this->setMaterialUnitStock( $ingredient[ 'REF_MATERIAL' ], $ingredient[ 'REF_UNIT' ], $quantity );
                $this->deleteOrderProductSummary( $order_product_id );
            }
        }

        return [
            'status'    =>  'failed',
            'message'   =>  __( 'Unable to find the requested recipe' )
        ];
    }

    /**
     * Delete material summary by order product
     * @param int order id
     * @param string status
     * @return array
     */
    public function deleteOrderProductSummary( $order_product_id, $status = 'consumption' )
    {
        $this->db->where( 'REF_ORDER_PRODUCT', $order_product_id )
            ->where( 'STATUS', $status )
            ->delete( store_prefix() . 'gastro_materials_usage_summary' );
        
        return [
            'status'    =>  'success',
            'message'   =>  __( 'The deletion has been made successfully.', 'raw-material' )
        ];
    }

    public function convertUnitFrom( $from, $to, float $quantity, $material_id )
    {
        $unitFrom   =   $this->getUnit( $from );
        $unitTo     =   $this->getUnit( $to );
        $baseUnit   =   $this->getGroupBaseUnit( $unitFrom[ 'REF_UNIT_GROUP' ] );

        if ( $unitFrom[ 'REF_UNIT_GROUP' ] !== $unitTo[ 'REF_UNIT_GROUP' ]) {
            return [
                'status'    =>  'failed',
                'message'   =>  __( 'Unable to convert unit from unit having different group', 'raw-material' )
            ];
        }

        /**
         * formula
         * ( ( total quantity * source unit value ) / destination unit value );
         */
        $converted  =   ( ( $quantity * floatval( $unitFrom[ 'UNIT_VALUE' ] ) ) / floatval( $unitTo[ 'UNIT_VALUE' ] ) );

        if ( $converted < 1 ) {
            return [
                'status'    =>  'failed',
                'message'   =>  __( 'The convertion cannot occurs as the quantity provided is not enough to produce at least 1 entity of the destination unit.', 'raw-material' )
            ];
        }

        $this->decreaseStockUnit( $material_id, $unitFrom[ 'ID' ], $quantity );
        $this->recordIngredientTracking( 'convert-out', $material_id, $unitFrom[ 'ID' ], $quantity );
        $this->increaseStockUnit( $material_id, $unitTo[ 'ID' ], $converted );
        $this->recordIngredientTracking( 'convert-in', $material_id, $unitTo[ 'ID' ], $converted );

        $material   =   $this->getMaterial( $material_id );

        nexo_log(
            __( 'Material Convertion', 'raw-material' ),
            sprintf(
                __( 'The user <strong>%s</strong> has made a material convertion for <strong>%s</strong> from <strong>%s</strong>(%s) to <strong>%s</strong>(%s).', 'raw-material' ),
                User::pseudo(),
                $material[ 'NAME' ],
                $unitFrom[ 'NAME' ],
                $quantity,
                $unitTo[ 'NAME' ],
                $converted
            )
        );

        /**
         * hopefully everything happened 
         * correctly without issue
         */
        return [
            'status'    =>  'success',
            'message'   =>  __( 'The conversion has occured correctly.', 'raw-material' )
        ];
    }

    public function getMaterialsSummary( $start = null, $end = null )
    {
        try {
            $startDateCarbon        =   Carbon::parse( $start );
            $endDateCarbon          =   Carbon::parse( $end );
        } catch( \Exception $e ) {
            $startDateCarbon    =   Carbon::parse( date_now() );
            $endDateCarbon      =   Carbon::parse( date_now() );
        }

        $startDateString    =   $startDateCarbon->startOfday()->toDateTimeString();    
        $endDateString      =   $endDateCarbon->endOfDay()->toDateTimeString();    

        $result             =   $this->db->select(
            '
            SUM( QUANTITY ) as QUANTITY,
            STATUS as STATUS,
            REF_UNIT as REF_UNIT,
            REF_MATERIAL as REF_MATERIAL,
            DATE_CREATION as DATE_CREATION
            '
        )
        ->from( store_prefix() . 'gastro_materials_usage_summary' )
        ->where( 'DATE_CREATION >=', $startDateString )
        ->where( 'DATE_CREATION <=', $endDateString )
        ->group_by([ 'STATUS', 'REF_UNIT', 'REF_MATERIAL', 'DATE_CREATION' ])
        ->get()
        ->result_array();

        return $result;
    }

    public function reset( $clearAll = false )
    {
        $groups     =   include_once( dirname( __FILE__ ) . '/../default.php' );

        if ( $clearAll ) {
            $this->db->query( 'TRUNCATE `' . $this->db->dbprefix . store_prefix() . 'gastro_materials_units`' );
            $this->db->query( 'TRUNCATE `' . $this->db->dbprefix . store_prefix() . 'gastro_materials_units_groups`');
            $this->db->query( 'TRUNCATE `' . $this->db->dbprefix . store_prefix() . 'gastro_materials`');
            $this->db->query( 'TRUNCATE `' . $this->db->dbprefix . store_prefix() . 'gastro_materials_units_stock`' );
            $this->db->query( 'TRUNCATE `' . $this->db->dbprefix . store_prefix() . 'gastro_materials_recipes_summary`');
            $this->db->query( 'TRUNCATE `' . $this->db->dbprefix . store_prefix() . 'gastro_materials_usage`');
            $this->db->query( 'TRUNCATE `' . $this->db->dbprefix . store_prefix() . 'gastro_materials_usage_summary`' );
            $this->db->query( 'TRUNCATE `' . $this->db->dbprefix . store_prefix() . 'gastro_materials_recipes`');
            $this->db->query( 'TRUNCATE `' . $this->db->dbprefix . store_prefix() . 'gastro_materials_recipes_items`');
            $this->db->query( 'TRUNCATE `' . $this->db->dbprefix . store_prefix() . 'gastro_materials_supplies`');
            $this->db->query( 'TRUNCATE `' . $this->db->dbprefix . store_prefix() . 'gastro_materials_supplies_items`');
        }

        foreach( $groups as $group ) {
            $this->db->insert( store_prefix() . 'gastro_materials_units_groups', [
                'AUTHOR'        =>  User::id(),
                'DATE_CREATION' =>  date_now(),
                'Date_MOD'      =>  date_now(),
                'NAME'          =>  $group[ 'name' ]
            ]);

            $group_id   =   $this->db->insert_id();

            foreach( $group[ 'units' ] as $unit ) {
                $data   =   [
                    'NAME'              =>  $unit[ 'name' ],
                    'UNIT_NAME'         =>  $unit[ 'unit_name' ],
                    'UNIT_VALUE'        =>  $unit[ 'unit_value' ],
                    'REF_UNIT_GROUP'    =>  $group_id,
                    'DATE_CREATION'     =>  date_now(),
                    'Date_MOD'          =>  date_now(),
                    'AUTHOR'            =>  User::id(),
                    'BASE_UNIT'         =>  @$unit[ 'default' ] === true ? 1 : 0
                ];

                $this->db->insert( store_prefix() . 'gastro_materials_units', $data );
            }

            foreach( $group[ 'materials' ] as $material ) {
                $this->db->insert( store_prefix() . 'gastro_materials', [
                    'AUTHOR'            =>  User::id(),
                    'DATE_CREATION'     =>  date_now(),
                    'DATE_MOD'          =>  date_now(),
                    'SKU'               =>  $material[ 'sku' ],
                    'NAME'              =>  $material[ 'name' ],
                    'REF_UNIT_GROUP'    =>  $group_id,
                ]);
            }
        }
    }

    /**
     * Get a specific recipe summary for a specific
     * time range provided
     * @param string start date
     * @param string end date
     * @return array
     */
    public function getRecipeSummary( $startDate = null, $endDate = null )
    {
        try {
            $startDateCarbon        =   Carbon::parse( $startDate );
            $endDateCarbon          =   Carbon::parse( $endDate );
        } catch( \Exception $e ) {
            $startDateCarbon    =   Carbon::parse( date_now() );
            $endDateCarbon      =   Carbon::parse( date_now() );
        }

        $startDateString    =   $startDateCarbon->startOfday()->toDateTimeString();    
        $endDateString      =   $endDateCarbon->endOfDay()->toDateTimeString();

        $result             =   $this->db->select(
            '
            SUM( QUANTITY ) as QUANTITY,
            STATUS as STATUS,
            REF_RECIPE as REF_RECIPE,
            REF_PRODUCT as REF_PRODUCT,
            DATE_CREATION as DATE_CREATION
            '
        )
        ->from( store_prefix() . 'gastro_materials_recipes_summary' )
        ->where( 'DATE_CREATION >=', $startDateString )
        ->where( 'DATE_CREATION <=', $endDateString )
        ->group_by([ 'STATUS', 'REF_RECIPE', 'REF_PRODUCT', 'DATE_CREATION' ])
        ->get()
        ->result_array();

        return $result;
    }

    public function defineConversionStatus( $conversion_type, $stock_unit, $status )
    {
        if ( $conversion_type === 'incoming' ) {
            $this->db->where( 'ID', $stock_unit )
                ->update( store_prefix() . 'gastro_materials_units_stock', [
                    'INCOMING_CONVERSION'   =>  $status === 'allow' ? 1 : 0
                ]);
        } else if ( $conversion_type === 'outgoing' ) {
            $this->db->where( 'ID', $stock_unit )
                ->update( store_prefix() . 'gastro_materials_units_stock', [
                    'OUTGOING_CONVERSION'   =>  $status === 'allow' ? 1 : 0
                ]);
        }

        return [
            'status'    =>  'success',
            'message'   =>  __( 'The operation was successful', 'raw-material' )
        ];
    }

    public function getSiblingUnitsStock( $material_id, $unit_id )
    {
        return $unitStock  =   $this->db->where( 'REF_MATERIAL', $material_id )
            ->where( 'REF_UNIT !=', $unit_id )
            ->get( store_prefix() . 'gastro_materials_units_stock' )
            ->result_array();
    }

    public function autoConvertIfPossible( $results )
    {
        collect( $results )->each( function( $result )  {
            if( $result[ 'depletion' ][ 'status' ] ) {
                collect( $result[ 'depletion' ][ 'data' ][ 'result' ] )->each( function( $deplete ) {
                    $data           =   $deplete[ 'data' ];
    
                    if ( isset( $data[ 'fromUnit' ] ) ) {
                        $this->convertUnitFrom( 
                            $data[ 'fromUnit' ][ 'ID' ], 
                            $data[ 'toUnit' ][ 'ID' ], 
                            $data[ 'totalToConvert' ], 
                            $data[ 'material' ][ 'ID' ] 
                        );
                    }
                });
            }
        });
    }

    public function trackStockAdjustment( $material_id, $unit_id, $quantity, $action, $description = '' )
    {
        $this->db->insert( store_prefix() . 'gastro_materials_adjustments', [
            'AUTHOR'    =>  User::id(),
            'ACTION'    =>  $action,
            'REF_MATERIAL'  =>  $material_id,
            'REF_UNIT'      =>  $unit_id,
            'DESCRIPTION'   =>  $description,
            'DATE_CREATION' =>  date_now(),
            'VALUE'         =>  $quantity
        ]);
    }
}